/**
 * \file: mspin_udp_helper.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN UDP Helper Functions
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
 * \copyright: (c) 2018 Bosch SoftTec GmbH
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_udp_helper.h"
#include "mspin_logging.h"

#include <arpa/inet.h>      //inet_ntop
#include <ifaddrs.h>        //ifaddrs
#include <errno.h>          //errno

void mspin_udp_freeBTName(mspin_udp_MessageParameter_t *pMessage)
{
    if (pMessage && pMessage->pBTName)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(pMessage=%p) deleting BT name '%s'...",
                __FUNCTION__, pMessage, pMessage->pBTName);
        free(pMessage->pBTName);
        pMessage->pBTName = NULL;
    }
}

MSPIN_ERROR mspin_udp_validateMessageParameters(MSPIN_UDP_MESSAGE_PARAMETER_t *pMessageParameters)
{
    if (!pMessageParameters)
    {
        mspin_log_printLn(eMspinVerbosityError,
                "%s() ERROR: pMessageParameters is NULL",
                __FUNCTION__);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    //Validate length of strings
    if ((strlen((const char *)pMessageParameters->manufacturer) > 64) || (strlen((const char *)pMessageParameters->vendor) > 64)
            || (strlen((const char *)pMessageParameters->model) > 64) || (strlen((const char *)pMessageParameters->version) > 12))
    {
        if (strlen((const char *)pMessageParameters->manufacturer) > 64)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Manufacturer name is too long. Only 64 bytes",
                    __FUNCTION__);
        }

        if (strlen((const char *)pMessageParameters->vendor) > 64)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Vendor name is too long. Only 64 bytes",
                    __FUNCTION__);
        }

        if (strlen((const char *)pMessageParameters->model) > 64)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Model name is too long. Only 64 bytes",
                    __FUNCTION__);
        }

        if (strlen((const char *)pMessageParameters->version) > 12)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: Version is too long. Only 12 bytes",
                    __FUNCTION__);
        }

        return MSPIN_ERROR_INVALID_PARAMETER;
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_udp_copyMessageParameters(MSPIN_UDP_MESSAGE_PARAMETER_t *pSourceMessage, mspin_udp_MessageParameter_t *pDestMessage)
{
    if (!pSourceMessage)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(srcMessage=%p, destMessage=%p) ERROR: Source message is NULL",
                __FUNCTION__, pSourceMessage, pDestMessage);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    if (!pDestMessage)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(srcMessage=%p, destMessage=%p) ERROR: Destination message is NULL",
                __FUNCTION__, pSourceMessage, pDestMessage);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    //Copy strings
    strcpy(pDestMessage->manufacturer, (const char *)pSourceMessage->manufacturer);
    strcpy(pDestMessage->vendor, (const char *)pSourceMessage->vendor);
    strcpy(pDestMessage->model, (const char *)pSourceMessage->model);
    strcpy(pDestMessage->version, (const char *)pSourceMessage->version);

    //Allocate memory for BT name and copy it, if set and not empty
    if (pSourceMessage->btName && (strlen((const char*)pSourceMessage->btName) > 0))
    {
        //Make sure the BT name is freed
        mspin_udp_freeBTName(pDestMessage);

        //Allocate memory for BT name
        pDestMessage->pBTName = (char*)malloc(strlen((const char*)pSourceMessage->btName) + 1);
        if (pDestMessage->pBTName)
        {
            strcpy(pDestMessage->pBTName, (const char*)pSourceMessage->btName);

            mspin_log_printLn(eMspinVerbosityDebug,
                 "%s(srcMessage=%p, destMessage=%p) Setting BT name '%s'",
                 __FUNCTION__, pSourceMessage, pDestMessage, pDestMessage->pBTName);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError,
                 "%s(srcMessage=%p, destMessage=%p) ERROR: Not enough memory for BT name",
                 __FUNCTION__, pSourceMessage, pDestMessage);

            return MSPIN_ERROR_MEM_ALLOC;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(srcMessage=%p, destMessage=%p) BT name not set or empty",
                __FUNCTION__, pSourceMessage, pDestMessage);
    }

    pDestMessage->tls = pSourceMessage->tls;
    pDestMessage->tcpPort = pSourceMessage->tcpPort;

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_udp_createUDPMessage(mspin_udp_MessageParameter_t *pMessage, char* udpMessage, unsigned int length,
                                       const char* hostString)
{
    int written = 0;
#ifdef MSPIN_UDP_MESSAGE_OLD_FORMAT
    written = snprintf(udpMessage, length, "%s://%s:%d", MSPIN_UDP_MESSAGE_PREFIX, hostString, tcpPort);
#else
    char identification[512];
    unsigned int version = 1;

    //Check message exists
    if (!pMessage)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(msg=%p, length=%d, hostString='%s') FATAL ERROR: pMessage is NULL",
                __FUNCTION__, pMessage, length, hostString);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    if (pMessage->tls || pMessage->pBTName)
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(msg=%p, length=%d, hostString='%s') TLS and/or BTName configured -> use version 2",
                __FUNCTION__, pMessage, length, hostString);
        version = 2;
    }

    snprintf(identification, sizeof(identification),
            "\"identification\": {\"manufacturer\":\"%s\", \"vendor\":\"%s\",\"model\":\"%s\", \"version\":\"%s\"}",
            pMessage->manufacturer, pMessage->vendor, pMessage->model, pMessage->version);

    if (version > 1)
    {
        //Add BT name if set. Otherwise omit parameter 'btname'
        if (pMessage->pBTName)
        {
            written = snprintf(udpMessage, length, "{\"version\" : %u, \"type\" : \"%s\", \"hostname\":\"%s\", \"port\":%d, \"btname\":\"%s\", %s}",
                    version, pMessage->tls ? "TLS" : "TCP", hostString, pMessage->tcpPort, pMessage->pBTName, identification);
        }
        else
        {
            written = snprintf(udpMessage, length, "{\"version\" : %u, \"type\" : \"%s\", \"hostname\":\"%s\", \"port\":%d, %s}",
                    version, pMessage->tls ? "TLS" : "TCP", hostString, pMessage->tcpPort, identification);
        }
    }
    else
    {
        written = snprintf(udpMessage, length, "{\"version\" : %u, \"hostname\":\"%s\", \"port\":%d, %s}",
                version, hostString, pMessage->tcpPort, identification);
    }
#endif //MSPIN_UDP_MESSAGE_OLD_FORMAT

    if (written >= (int)length)
    {
        mspin_log_printLn(eMspinVerbosityFatal,
                "%s(msg=%p, length=%d, hostString='%s') ERROR: UDP message truncated because too long",
                __FUNCTION__, pMessage, length, hostString);
        return MSPIN_ERROR_GENERAL;
    }

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_udp_getBroadcastAddress(const U8 *interface, in_addr_t *pIPAddr, in_addr_t *pNetmask)
{
    MSPIN_ERROR result = MSPIN_ERROR_NOT_READY;
    struct ifaddrs *pIfAddrStruct = NULL;
    struct ifaddrs *pIfa = NULL;
    void *tmpAddrPtr = NULL;
    char hostString[NI_MAXHOST];
    char maskString[NI_MAXHOST];

    //Check parameter
    if (!pIPAddr)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(interface='%s') ERROR: pIPAddr is NULL", __FUNCTION__, interface);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    if (!pNetmask)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(interface='%s') ERROR: pNetmask is NULL", __FUNCTION__, interface);
        return MSPIN_ERROR_NULL_HANDLE;
    }

    //Reset values
    *pIPAddr = 0;
    *pNetmask = 0;

    //Get all interfaces
    if (-1 == getifaddrs(&pIfAddrStruct))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(interface='%s') ERROR: Failed to get if address with '%s'(%d)",
                __FUNCTION__, interface, strerror(errno), errno);
        return MSPIN_ERROR_NOT_READY;
    }

    //Go through all interfaces and check if the specified interface is in the list
    for (pIfa = pIfAddrStruct; pIfa != NULL; pIfa = pIfa->ifa_next)
    {
        if (!pIfa->ifa_addr)
        {
            continue;
        }

        if (pIfa->ifa_addr->sa_family == AF_INET)
        {
            //Check only IP v4 interfaces
            tmpAddrPtr=&((struct sockaddr_in *)pIfa->ifa_addr)->sin_addr;
            char addressBuffer[INET_ADDRSTRLEN];
            inet_ntop(AF_INET, tmpAddrPtr, addressBuffer, INET_ADDRSTRLEN);

            if (0 == strcmp((const char*)interface, pIfa->ifa_name))
            {
                mspin_log_printLn(eMspinVerbosityDebug, "%s(interface='%s') matching interface '%s' found with IP v4 addr='%s'",
                                __FUNCTION__, interface, pIfa->ifa_name, addressBuffer);

                *pIPAddr = ((struct sockaddr_in *)pIfa->ifa_addr)->sin_addr.s_addr;
                *pNetmask = ((struct sockaddr_in *)pIfa->ifa_netmask)->sin_addr.s_addr;

                result = MSPIN_SUCCESS;

                //Print found IP address and
                if (0 == getnameinfo(pIfa->ifa_addr, sizeof(struct sockaddr_in), hostString, NI_MAXHOST, NULL, 0, NI_NUMERICHOST))
                {
                    if (0 == getnameinfo(pIfa->ifa_netmask, sizeof(struct sockaddr_in), maskString, NI_MAXHOST, NULL, 0, NI_NUMERICHOST))
                    {
                        mspin_log_printLn(eMspinVerbosityDebug, "%s(interface='%s') IP addr=%s and mask=%s",
                                __FUNCTION__, interface, hostString, maskString);
                    }
                    else
                    {
                        mspin_log_printLn(eMspinVerbosityDebug, "%s(interface='%s') ERROR: Failed to get netmask for interface='%s'",
                                __FUNCTION__, pIfa->ifa_name);
                    }
                }
                else
                {
                    mspin_log_printLn(eMspinVerbosityDebug, "%s(interface='%s') ERROR: Failed to get IPv4 address for interface='%s'",
                            __FUNCTION__, pIfa->ifa_name);
                }
            }
            else
            {
                mspin_log_printLn(eMspinVerbosityVerbose, "%s(interface='%s') other interface '%s' found with IP v4 addr='%s'",
                                                __FUNCTION__, interface, pIfa->ifa_name, addressBuffer);
            }
        }
//        else if (pIfa->ifa_addr->sa_family == AF_INET6)
//        {
//            //It is a valid IP6 Address
//            tmpAddrPtr=&((struct sockaddr_in6 *)pIfa->ifa_addr)->sin6_addr;
//            char addressBuffer[INET6_ADDRSTRLEN];
//            inet_ntop(AF_INET6, tmpAddrPtr, addressBuffer, INET6_ADDRSTRLEN);
//
//            mspin_log_printLn(eMspinVerbosityDebug, "%s(interface='%s') interface '%s' found with IP v6 addr='%s'",
//                            __FUNCTION__, interface, pIfa->ifa_name, addressBuffer);
//        }
    }

    if (pIfAddrStruct)
    {
        freeifaddrs(pIfAddrStruct);
    }

    if (MSPIN_SUCCESS != result)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(interface='%s') ERROR: Interface not found",
                __FUNCTION__, interface);
    }

    return result;
}
